<?php
/**
 * LICENSE
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@smartsend.io so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade the plugin to newer
 * versions in the future. If you wish to customize the plugin for your
 * needs please refer to https://www.smartsend.io
 *
 * @author      Smart Send ApS
 * @copyright   Copyright (c) Smart Send ApS (https://www.smartsend.io)
 * @license     https://smartsend.io/license
 * @url         https://smartsend.io/
 * @class       SmartsendLabelController
 * @folder      smartsend/controllers/admin/SmartsendLabelController.php
 *
 */

class SmartsendLabelController extends ModuleAdminController
{

    protected $shipment = null;
    protected $logger = null;
    protected $order;
    protected $return;

    public function __construct($order, $return = false)
    {
        $this->order = $order;
        $this->return = $return;

        parent::__construct();
    }

    public function createShipmentAndGenerateLabel($type = false, $orderId = 0)
    {
        $response = $this->createLabel($type);

        if (isset($response['data'])) {
            $responseData = $response['data'];

            $id_order_carrier = $this->order->getIdOrderCarrier(($orderId > 0) ? $orderId : Tools::getValue('id_order'));
            $order_carrier = new OrderCarrier((int)$id_order_carrier);

            foreach ($responseData->parcels as $parcels) {
                if ($this->order->getWsShippingNumber() != $parcels->tracking_code) {
                    $this->order->setWsShippingNumber($parcels->tracking_code);
                    $this->order->shipping_number = $parcels->tracking_code;//Required for in_transit email to work on PS 1.7
                    $this->order->update();

                    // Send mail to customer
                    if (Configuration::get('SMARTSEND_SHIPMENT_EMAIL', null)) {

                        if (version_compare(_PS_VERSION_, '1.7') < 0) {
                            // PrestaShop 1.6
                            $customer = new Customer((int)$this->order->id_customer);
                            $carrier = new Carrier((int)$this->order->id_carrier, $this->order->id_lang);
                            if (!Validate::isLoadedObject($customer)) {
                                throw new PrestaShopException('Can\'t load Customer object');
                            }
                            if (!Validate::isLoadedObject($carrier)) {
                                throw new PrestaShopException('Can\'t load Carrier object');
                            }
                            $templateVars = array(
                                '{followup}'        => $parcels->tracking_link,
                                '{firstname}'       => $customer->firstname,
                                '{lastname}'        => $customer->lastname,
                                '{id_order}'        => $this->order->id,
                                '{shipping_number}' => $this->order->shipping_number,
                                '{order_name}'      => $this->order->getUniqReference()
                            );
                            @Mail::Send((int)$this->order->id_lang, 'in_transit',
                                Mail::l('Package in transit', (int)$this->order->id_lang), $templateVars,
                                $customer->email, $customer->firstname . ' ' . $customer->lastname, null, null, null,
                                null,
                                _PS_MAIL_DIR_, true, (int)$this->order->id_shop);
                        } else {
                            // PrestaShop 1.7
                            $order_carrier->sendInTransitEmail($this->order);
                        }
                    }

                    Hook::exec('actionSmartSendAdminOrdersTrackingInfoUpdate',
                        array(
                            'order' => $this->order,
                        ),
                        $id_module = null,
                        $array_return = false,
                        $check_exceptions = true,
                        $use_push = false,
                        $id_shop = $this->order->id_shop,
                        $chain = true
                    );
                }
            }

            // Add Order Message with info about created label and tracking number
            if (Configuration::get('SMARTSEND_ORDER_MESSAGE', false) && $this->order->getWsShippingNumber()) {
                $this->addOrderPrivateMessage($this->order->getWsShippingNumber());
            }

            // Change order status
            $new_order_status = (int) Configuration::get('SMARTSEND_ORDER_STATUS', null);
            $order_status = (int)$this->order->getCurrentOrderState()->id;
            if ($new_order_status && $new_order_status != $order_status) {
                $this->updateOrderStatus();
            }

            if (isset($responseData->pdf->link)) {
                $pdf_link = $responseData->pdf->link;
                $this->savePDFDetails($pdf_link, ($type == 'return_label'));

                if ($type == 'return_label') {
                    $tmp_translated_success = $this->transInline(
                        "Order #%s: Return Shipping label created by Smart Send.",
                        array(),
                        'Admin.Orderscustomers.Notification'
                    )
                        . ' <a href="%s" target="_blank">'
                        . $this->transInline(
                            "Download Return Shipping Label.",
                            array(),
                            'Admin.Orderscustomers.Notification'
                        )
                        . '</a>';
                } else {
                    $tmp_translated_success = $this->transInline(
                        "Order #%s: Shipping label created by Smart Send.",
                        array(),
                        'Admin.Orderscustomers.Notification'
                    )
                        . ' <a href="%s" target="_blank">'
                        . $this->transInline(
                            "Download Shipping Label.",
                            array(),
                            'Admin.Orderscustomers.Notification'
                        )
                        . '</a>';
                }
                $this->confirmations[] = sprintf($tmp_translated_success, $this->order->reference, $pdf_link);
                return $responseData->shipment_id;
            } else {
                $tmp_translated_error = $this->transInline(
                    "Order #%s: No PDF returned from Smart Send",
                    array(),
                    'Admin.Orderscustomers.Notification'
                );
                $this->errors[] = sprintf($tmp_translated_error, $this->order->reference);
                return false;
            }

            return true;
        } elseif (isset($response['errors'])) {
            $this->errors[] = $response['errors'];
            return false;
        } else {
            $tmp_translated_error = $this->transInline(
                "Order #%s: Unknown error when trying to generate shipping label",
                array(),
                'Admin.Orderscustomers.Notification'
            );
            $this->errors[] = sprintf($tmp_translated_error, $this->order->reference);
            return false;
        }
    }

    /**
     * Add private Order Message with info about created label and tracking number
     *
     * @param $tracking_number
     * @return bool
     */
    private function addOrderPrivateMessage($tracking_number)
    {
        $customer = new Customer((int)$this->order->id_customer);
        $employee = new Employee((int)$this->context->cookie->id_employee);

        $customer_thread = new CustomerThread();
        $customer_thread->id_contact = 0;
        $customer_thread->id_customer = (int)$this->order->id_customer;
        $customer_thread->id_shop = (int)$this->context->shop->id;
        $customer_thread->id_order = (int)$this->order->id;
        $customer_thread->id_lang = (int)$this->context->language->id;
        $customer_thread->email = $customer->email;
        $customer_thread->status = 'open';
        $customer_thread->token = Tools::passwdGen(12);
        $customer_thread->add();

        $customer_message = new CustomerMessage();
        $customer_message->id_customer_thread = $customer_thread->id;
        $customer_message->message = nl2br('Shipping Label created by Smart Send Tracking Numbers: '
            . $tracking_number);
        $customer_message->cfirstname = $customer->firstname;
        $customer_message->clastname = $customer->lastname;
        $customer_message->efirstname = $employee->firstname;
        $customer_message->elastname = $employee->lastname;
        $customer_message->private = 1;//Display to customer? 0: Yes 1: No
        $customer_message->id_employee = $this->context->cookie->id_employee;

        if ($customer_message->validateFields(false)) {
            $customer_message->add();
        }
        return true;
    }

    private function savePDFDetails($pdf_link = null, $return = false)
    {
        include_once _PS_MODULE_DIR_ . 'smartsend/smartsendshipmentlabels.php';
        $smartsend_method = new SmartSendMethodsClass();
        $rule_id = $smartsend_method->getShippingMethodByCarrierId($this->order->id_carrier);
        $shipping_rule_details = new SmartSendMethodsClass($rule_id);

        $shipping_method_id = explode('_', $shipping_rule_details->shipping_method);
        $shipping_carrier = $shipping_method_id[0];

        if ($this->order->id && $shipping_carrier) {
            $smartsend_label = new SmartSendShipmentLabelsClass();
            if ($smartsend_label->getShipmentLabelIdByOrderId($this->order->id, $shipping_carrier)) {
                $smartsend_shipment = new SmartSendShipmentLabelsClass(
                    $smartsend_label->getShipmentLabelIdByOrderId($this->order->id, $shipping_carrier)
                );
            } else {
                $smartsend_shipment = new SmartSendShipmentLabelsClass();
            }
        } else {
            $smartsend_shipment = new SmartSendShipmentLabelsClass();
        }

        $data = array();
        $data['order_id'] = $this->order->id;
        $data['shipping_method'] = $shipping_carrier;

        if ($return) {
            $data['return_pdf_link'] = $pdf_link;
        } else {
            $data['pdf_link'] = $pdf_link;
        }

        $smartsend_shipment->copyFromPost($data);
        if ($smartsend_shipment->validateFields(false)) {
            $smartsend_shipment->save();
        }
        return true;
    }

    private function createLabel($type)
    {
        if (!$this->getShippingMethod()) {
            if ($this->return) {
                $error = $this->transInline(
                    "Order %s: There is no Smart Send return shipping method associated with this order (MethodId=%s)",
                    array(),
                    'Admin.Orderscustomers.Notification'
                );
            } else {
                $error = $this->transInline(
                    "Order %s: There is no Smart Send shipping method associated with this order (MethodId=%s)",
                    array(),
                    'Admin.Orderscustomers.Notification'
                );
            }

            return array(
                "errors" => sprintf($error, $this->order->reference, $this->order->id_carrier),
            );
        }

        $this->mapRequestToShipment($type);
        $response = $this->doRequest();

        // Format error to include the order reference
        if (isset($response['error']) && (($response['error'] != null) || (!empty($response['error'])))) {
            $response['errors'] = sprintf("Order %s: ".$response['error'], $this->order->reference);
        }

        if (empty($response)) {
            $error = $this->transInline(
                "Unknown error from label request",
                array(),
                'Admin.Orderscustomers.Notification'
            );
            $response['errors'] = sprintf("Order %s: ".$error, $this->order->reference);
        }

        return $response;
    }

    /**
     * Do rate request and handle errors
     *
     * @return array contains data-field if successful and errors if failed
     */
    protected function doRequest()
    {
        include_once _PS_MODULE_DIR_ . 'smartsend/classes/class.api.php';
        $smartsend_api = new SmartSendShippingApi();
        return $smartsend_api->generateShippingLabel($this->shipment);
    }

    private function getShippingMethod()
    {
        include_once _PS_MODULE_DIR_ . 'smartsend/smartsendmethods.php';
        $smartsend_method = new SmartSendMethodsClass();
        $rule_id = $smartsend_method->getShippingMethodByCarrierId($this->order->id_carrier);
        $shipping_rule_details = new SmartSendMethodsClass($rule_id);

        if ($this->return) {
            if (empty($shipping_rule_details->return_method)) {
                return false;
            } else {
                return $shipping_rule_details->return_method;
            }
        } else {
            if (empty($shipping_rule_details->shipping_method)) {
                return false;
            } else {
                return $shipping_rule_details->shipping_method;
            }
        }
    }


    private function mapRequestToShipment($type)
    {
        include_once _PS_MODULE_DIR_ . 'smartsend/lib/Smartsend/Api.php';
        include_once _PS_MODULE_DIR_ . 'smartsend/classes/class.agent.php';

        $currency = new CurrencyCore($this->order->id_currency);
        $delivery_address = new Address($this->order->id_address_delivery);
        $customer = new Customer($delivery_address->id_customer);
        $country = $country = new Country($delivery_address->id_country);

        $shipping_method = $this->getShippingMethod();

        $currency_code = $currency->iso_code;

        if ($type == 'return_label') {
            $smartsend_method = new SmartSendMethodsClass();
            $rule_id = $smartsend_method->getShippingMethodByCarrierId($this->order->id_carrier);
            $shipping_rule_details = new SmartSendMethodsClass($rule_id);
            $shipping_return_method = $shipping_rule_details->return_method;

            $shipping_method_id = explode('_', $shipping_return_method);

            $shipping_carrier = $shipping_method_id[0];
            $shipping_method = $shipping_method_id[1];
        } else {
            $shipping_method_id = explode('_', $shipping_method);
            $shipping_carrier = $shipping_method_id[0];
            $shipping_method = $shipping_method_id[1];
        }

        //New shipment model
        $shipment = new \Smartsend\Models\Shipment();
        $receiver = new \Smartsend\Models\Shipment\Receiver();
        $receiver->setInternalId($this->order->id_address_delivery)
            ->setInternalReference($this->order->id_address_delivery)
            ->setCompany($delivery_address->company)
            ->setNameLine1($delivery_address->firstname)
            ->setNameLine2($delivery_address->lastname)
            ->setAddressLine1($delivery_address->address1)
            ->setAddressLine2($delivery_address->address2)
            ->setPostalCode($delivery_address->postcode)
            ->setCity($delivery_address->city)
            ->setCountry($country->iso_code)
            ->setSms($delivery_address->phone_mobile ? $delivery_address->phone_mobile : $delivery_address->phone)
            ->setEmail($customer->email);

        // Add the receiver to the shipment
        $shipment->setReceiver($receiver);

        // Add an agent (pickup point) to the shipment
        $smartsend_agent = new SmartSendAgentClass();
        $agent_points = $smartsend_agent->getAgentDetailsByOrderId($this->order->id);
        if (isset($agent_points['agent_no']) && $type != 'return_label') {
            $agent = new Smartsend\Models\Shipment\Agent();
            $agent->setInternalId($agent_points['id'])
                ->setInternalReference($agent_points['id'])
                ->setAgentNo($agent_points['agent_no'])
                ->setCompany($agent_points['company'])
                ->setNameLine1(null)
                ->setNameLine2(null)
                ->setAddressLine1($agent_points['address_line1'])
                ->setAddressLine2(null)
                ->setPostalCode($agent_points['postal_code'])
                ->setCity($agent_points['city'])
                ->setCountry($agent_points['country'])
                ->setSms(null)
                ->setEmail(null);

            // Add the agent to the shipment
            $shipment->setAgent($agent);
        }

        // Create the parcels array
        $parcels = array();

        // Add the items that are send in the first parcel
        $ordered_items = $this->order->getProducts();
        $items = array();
        $package_weight = 0;
        $parcel_amt_incl_tax = 0;
        $parcel_amt_excl_tax = 0;
        $parcel_tax_amt = 0;

        foreach ($ordered_items as $orderItem) {
            $qty = $orderItem['product_quantity'];
            $item_unit_tax = round(($orderItem['unit_price_tax_incl'] - $orderItem['unit_price_tax_excl']), 2);

            $itemObj = new \Smartsend\Models\Shipment\Item();
            $itemObj->setInternalId($orderItem['id_order_detail'])
                ->setInternalReference($orderItem['id_order_detail'])
                ->setSku($orderItem['product_reference'])
                ->setName($orderItem['product_name'])
                ->setDescription(null)
                ->setHsCode(null)
                ->setImageUrl(null)
                ->setUnitWeight($orderItem['weight'] > 0 ? $orderItem['weight'] : null)
                ->setUnitPriceExcludingTax($orderItem['unit_price_tax_excl'])
                ->setUnitPriceIncludingTax($orderItem['unit_price_tax_incl'])
                ->setQuantity($qty)
                ->setTotalPriceExcludingTax($orderItem['total_price'])
                ->setTotalPriceIncludingTax($orderItem['total_wt'])
                ->setTotalTaxAmount($item_unit_tax * $qty);

            $parcel_amt_excl_tax += ($orderItem['total_price']);
            $parcel_amt_incl_tax += ($orderItem['total_wt']);
            $parcel_tax_amt += ($item_unit_tax * $qty);
            $package_weight += ($orderItem['weight'] * $qty);

            $items[] = $itemObj;
        }

        // Create a parcel containing the items just defined
        $parcel = new \Smartsend\Models\Shipment\Parcel();
        $parcel->setInternalId($this->order->delivery_number ?
            $this->order->delivery_number : $this->order->id)
            ->setInternalReference($this->order->delivery_number ?
                $this->order->delivery_number : $this->order->reference)
            ->setWeight($package_weight > 0 ? $package_weight : null)
            ->setHeight(null)
            ->setWidth(null)
            ->setLength(null)
            ->setFreetext(null)
            ->setItems($items)// Alternatively add each item using $parcel->addItem(Item $item)
            ->setTotalPriceExcludingTax($parcel_amt_excl_tax)
            ->setTotalPriceIncludingTax($parcel_amt_incl_tax)
            ->setTotalTaxAmount($parcel_tax_amt);
        $parcels[] = $parcel;

        // Create services
        $services = new \Smartsend\Models\Shipment\Services();
        $services->setSmsNotification($delivery_address->phone)
            ->setEmailNotification($customer->email);

        // Add final parameters to shipment
        $shipment->setInternalId($this->order->id)
            ->setInternalReference($this->order->reference)
            ->setShippingCarrier($shipping_carrier)
            ->setShippingMethod($shipping_method)
            ->setShippingDate(date('Y-m-d'))
            ->setParcels($parcels)// Alternatively add each parcel using $shipment->addParcel(Parcel $parcel);
            ->setServices($services)
            ->setSubtotalPriceExcludingTax($this->order->total_products)
            ->setSubtotalPriceIncludingTax($this->order->total_products_wt)
            ->setTotalPriceExcludingTax($this->order->total_paid_tax_excl)
            ->setTotalPriceIncludingTax($this->order->total_paid_tax_incl)
            ->setShippingPriceExcludingTax($this->order->total_shipping_tax_excl)
            ->setShippingPriceIncludingTax($this->order->total_shipping_tax_incl)
            ->setTotalTaxAmount($this->order->total_paid_tax_incl - $this->order->total_paid_tax_excl)
            ->setCurrency($currency_code);

        // Implement filter to change the parameters used for
        $filteredShipment = Hook::exec(
            'filterSmartSendShipmentParameters',
            array(
                'object' => $shipment
            ),
            $id_module = null,
            $array_return = false,
            $check_exceptions = true,
            $use_push = false,
            $id_shop = null,
            $chain = true
        );
        if (!empty($filteredShipment['object'])) {
            $shipment = $filteredShipment['object'];
        }

        $this->shipment = $shipment;
    }

    /**
     * Change order status, add a new entry in order history and send an e-mail to the customer if needed
     *
     * @return void
     */
    protected function updateOrderStatus()
    {
        $edit_allowed = false;
        $version = '1.7.0.0';
        if (version_compare(_PS_VERSION_, $version) < 0) {
            if ($this->tabAccess['edit'] === '1') {
                $edit_allowed = true;
            }
        } else {
            if ($this->access('edit')) {
                $edit_allowed = true;
            }
        }
        if ($edit_allowed) {
            $new_order_status = Configuration::get('SMARTSEND_ORDER_STATUS', null);
            $order_state = new OrderState($new_order_status);
            //$order_state = new OrderState(OrderState::FLAG_DELIVERY);

            if (!Validate::isLoadedObject($order_state)) {
                $error = 'The new order status is invalid.';
                $this->errors[] = $this->transInline($error, array(), 'Admin.Orderscustomers.Notification');
            } else {
                // Create new OrderHistory
                $history = new OrderHistory();
                $history->id_order = $this->order->id;
                $history->id_employee = (int)$this->context->employee->id;

                $use_existings_payment = false;
                if (!$this->order->hasInvoice()) {
                    $use_existings_payment = true;
                }
                $history->changeIdOrderState((int)$order_state->id, $this->order, $use_existings_payment);

                $carrier = new Carrier($this->order->id_carrier, $this->order->id_lang);
                $templateVars = array();
                $ps_os_shipping = Configuration::get('PS_OS_SHIPPING');
                if ($history->id_order_state == $ps_os_shipping && $this->order->shipping_number) {
                    $val = str_replace('@', $this->order->shipping_number, $carrier->url);
                    $templateVars = array('{followup}' => $val);
                }

                // Save all changes
                if ($history->addWithemail(true, $templateVars)) {
                    // synchronizes quantities if needed..
                    if (Configuration::get('PS_ADVANCED_STOCK_MANAGEMENT')) {
                        foreach ($this->order->getProducts() as $product) {
                            if (StockAvailable::dependsOnStock($product['product_id'])) {
                                StockAvailable::synchronize($product['product_id'], (int)$product['id_shop']);
                            }
                        }
                    }
                } else {
                    $error = 'An error occurred while changing order status';
                    $this->errors[] = $this->transInline($error, array(), 'Admin.Orderscustomers.Notification');
                }
            }
        } else {
            $error = 'You do not have permission to edit this.';
            $this->errors[] = $this->transInline($error, array(), 'Admin.Notifications.Error');
        }
    }

    private function transInline($error_msg, $data, $class)
    {
        $version = '1.7.0.0';
        if (version_compare(_PS_VERSION_, $version) < 0) {
            return $this->l($error_msg, $data, $class);
        } else {
            return $this->trans($error_msg, $data, $class);
        }
    }
}
