<?php
/**
 * NOTICE OF LICENSE
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@smartsend.io so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade the plugin to newer
 * versions in the future. If you wish to customize the plugin for your
 * needs please refer to https://www.smartsend.io
 *
 * @author    Smart Send <support@smartsend.io>
 * @copyright Smart Send
 * @license   http://www.gnu.org/licenses/gpl-3.0.html  GNU General Public License v3.0
 */

use PrestaShop\PrestaShop\Core\Grid\Definition\Factory\AbstractGridDefinitionFactory;
use PrestaShop\PrestaShop\Core\Grid\Action\Bulk\Type\SubmitBulkAction;
use PrestaShop\PrestaShop\Core\Grid\Action\Bulk\BulkActionCollection;

if (!defined('_PS_VERSION_')) {
    exit;
}

include_once _PS_MODULE_DIR_ . 'smartsend/smartsendmethods.php';
include_once _PS_MODULE_DIR_ . 'smartsend/classes/class.api.php';

class Smartsend extends CarrierModule
{
    protected $config_form = false;

    public function __construct()
    {
        $this->name = 'smartsend';
        $this->tab = 'shipping_logistics';
        $this->version = '1.1.2';
        $this->author = 'Smart Send';
        $this->need_instance = 1;

        /**
         * Set $this->bootstrap to true if your module is compliant with bootstrap (PrestaShop 1.6)
         */
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('Smart Send');
        $this->description = $this->l('Complete shipping solution. Automate your webshops shipping using Smart Send');

        $this->confirmUninstall = $this->l('Are you sure you want to uninstall Smart Send? This will remove all data like settings, selected pick-up points and labels!');

        $this->ps_versions_compliancy = array('min' => '1.7.6', 'max' => _PS_VERSION_);
    }

    /**
     * Don't forget to create update methods if needed:
     * http://doc.prestashop.com/display/PS16/Enabling+the+Auto-Update
     */
    public function install()
    {
        if (extension_loaded('curl') == false) {
            $this->_errors[] = $this->l('You have to enable the cURL extension on your server to install this module');
            return false;
        }



        // No carrier will be added
        //$carrier = $this->addCarrier();
        //$this->addZones($carrier);
        //$this->addGroups($carrier);
        //$this->addRanges($carrier);
        Configuration::updateValue('SMARTSEND_LIVE_MODE', false);


        // Add Shipping tab
        if (!$this->installSmartsendShippingTab()) {
            $this->_errors[] = Tools::displayError($this->l(' An error occurred while creating Smartsend Shipping tab.'));
            return false;
        }

        // Add Smartsend Action tab
        if (!$this->installSmartsendActionTab()) {
            $this->_errors[] = Tools::displayError($this->l(' An error occurred while creating Smartsend Action tab.'));
            return false;
        }

        // Add Label tab
        if (!$this->installSmartsendLabelTab()) {
            $this->_errors[] = Tools::displayError($this->l(' An error occurred while creating Smartsend Label tab.'));
            return false;
        }

        // Add Combine Label tab
        if (!$this->installSmartsendCombineLabelTab()) {
            $this->_errors[] = Tools::displayError($this->l(' An error occurred while creating Smartsend Combine Label tab.'));
            return false;
        }

        include(dirname(__FILE__) . '/sql/install.php');
        return parent::install() && $this->installDB() &&
            $this->registerHook('header') &&
            $this->registerHook('displayBackOfficeHeader') &&
            $this->registerHook('actionCarrierUpdate') &&
            $this->registerHook('actionCarrierProcess') &&
            $this->registerHook('displayBeforeCarrier') &&
			$this->registerHook('displayAdminOrderMain') &&
			$this->registerHook('displayBackOfficeFooter') &&
			$this->registerHook('displayAdminAfterHeader') &&
            $this->registerHook('actionValidateOrder');
    }

    /*
     * Add a Shipping tab to configure the module.
     * @return boolean
     */
    public function installSmartsendShippingTab()
    {
        $idTab = Tab::getIdFromClassName('SmartsendShipping');
        if ($idTab) {
            // Tab already found
            return false;
        }

        $tab = new Tab();
        $tab->class_name = 'SmartsendShipping';
        $tab->module = $this->name;
        $tab->id_parent = 0;
        $tab->active = 0;
        $languages = Language::getLanguages(false);
        foreach ($languages as $lang) {
            $tab->name[$lang['id_lang']] = 'Smart Send Shipping'; //Do not translate
        }
        $tab->name[(int)(Configuration::get('PS_LANG_DEFAULT'))] = 'SmartsendShipping'; //translate
        $tab->save();

        return true;
    }

    /*
     * Add a Label tab to configure the module.
     * @return boolean
     */
    public function installSmartsendLabelTab()
    {
        $idTab = Tab::getIdFromClassName('SmartsendLabel');
        if ($idTab) {
            // Tab already found
            return false;
        }

        $tab = new Tab();
        $tab->class_name = 'SmartsendLabel';
        $tab->module = $this->name;
        $tab->id_parent = 0;
        $tab->active = 0;
        $languages = Language::getLanguages(false);
        foreach ($languages as $lang) {
            $tab->name[$lang['id_lang']] = 'Generate Label Smart Send Shipping'; //Do not translate
        }
        $tab->name[(int)(Configuration::get('PS_LANG_DEFAULT'))] = 'SmartsendLabel'; //translate
        $tab->save();

        return true;
    }

    /*
     * Add a Combine Label tab to configure the module.
     * @return boolean
     */
    public function installSmartsendCombineLabelTab()
    {
        $idTab = Tab::getIdFromClassName('SmartsendCombineLabel');
        if ($idTab) {
            // Tab already found
            return false;
        }

        $tab = new Tab();
        $tab->class_name = 'SmartsendCombineLabel';
        $tab->module = $this->name;
        $tab->id_parent = 0;
        $tab->active = 0;
        $languages = Language::getLanguages(false);
        foreach ($languages as $lang) {
            $tab->name[$lang['id_lang']] = 'Combine Label Smart Send Shipping'; //Do not translate
        }
        $tab->name[(int)(Configuration::get('PS_LANG_DEFAULT'))] = 'SmartsendCombineLabel'; //translate
        $tab->save();

        return true;
    }

    /*
     * Add an Action tab to configure the module.
     * @return boolean
     */
    public function installSmartsendActionTab()
    {
        $idTab = Tab::getIdFromClassName('SmartsendAction');
        if ($idTab) {
            // Tab already found
            return false;
        }

        $tab = new Tab();
        $tab->class_name = 'SmartsendAction';
        $tab->module = $this->name;
        $tab->id_parent = 0;
        $tab->active = 0;
        $languages = Language::getLanguages(false);
        foreach ($languages as $lang) {
            $tab->name[$lang['id_lang']] = 'Smart Send Action'; //Do not translate
        }
        $tab->name[(int)(Configuration::get('PS_LANG_DEFAULT'))] = 'SmartsendAction'; //translate
        $tab->save();

        return true;
    }

    /**
     * This function creates new table that is required for saving the shipping methods
     */
    public function installDB()
    {
        $return = true;
        $return &= Db::getInstance()->execute('
            CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . 'smartsend_shipping` (
                `smethod_id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
                `id_carrier` INT UNSIGNED NOT NULL,
                `shipping_method` VARCHAR(100) default "",
                `return_method` VARCHAR(100) default "",
                `current_version` tinyint(1) NOT NULL DEFAULT 0,
                PRIMARY KEY (`smethod_id`)
            ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;
            CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . 'smartsend_shipment_labels` (
                `slabel_id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
                `order_id` INT UNSIGNED NOT NULL,
                `shipping_method` VARCHAR(100) NOT NULL,
                `pdf_link` VARCHAR(2083) default "",
                `return_pdf_link` VARCHAR(2083) default "",
                PRIMARY KEY (`slabel_id`)
            ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;
            CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . 'smartsend_agent` (
                `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
                `id_carrier` INT UNSIGNED NOT NULL,
                `cart_id` INT UNSIGNED NOT NULL,
                `order_id` INT UNSIGNED NOT NULL,
                `store` VARCHAR(100) NOT NULL,
                `agent_no` INT UNSIGNED NOT NULL,
                `company` VARCHAR(100) NOT NULL,
                `address_line1` VARCHAR(200) NOT NULL,
                `address_line2` VARCHAR(200) NOT NULL,
                `city` VARCHAR(200) NOT NULL,
                `postal_code` INT UNSIGNED NOT NULL,
                `country` VARCHAR(100) NOT NULL,
                PRIMARY KEY (`id`)
            ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;');

        return $return;
    }

    /**
     * This function removes the installed tables and other related things when uninstalling the module.
     */
    public function uninstall()
    {
        $version = '1.7';
        Configuration::deleteByName('SMARTSEND_LIVE_MODE');

        include(dirname(__FILE__) . '/sql/uninstall.php');

        if (version_compare(_PS_VERSION_, $version) < 0) {
            // Remove Smartsend Shipping tab
            if (!$this->uninstallSmartsendShippingTab()) {
                $message = ' An error occurred during uninstalling Smartsend Shipping tab.';
                $this->_errors[] = Tools::displayError($this->l($message));
                return false;
            }
            // Remove Smartsend Action tab
            if (!$this->uninstallSmartsendActionTab()) {
                $message = ' An error occurred during uninstalling Smartsend Action tab.';
                $this->_errors[] = Tools::displayError($this->l($message));
                return false;
            }
            // Remove Smartsend Label tab
            if (!$this->uninstallSmartsendLabelTab()) {
                $message = ' An error occurred during uninstalling Smartsend Label tab.';
                $this->_errors[] = Tools::displayError($this->l($message));
                return false;
            }
            // Remove Smartsend Label tab
            if (!$this->uninstallSmartsendCombineLabelTab()) {
                $message = ' An error occurred during uninstalling Smartsend Combine Label tab.';
                $this->_errors[] = Tools::displayError($this->l($message));
                return false;
            }
        }

        return $this->uninstallDB() && parent::uninstall();
    }

    public function uninstallDB()
    {
        return Db::getInstance()->execute('DROP TABLE IF EXISTS `' . _DB_PREFIX_ . 'smartsend_shipment_labels`') &&
            Db::getInstance()->execute('DROP TABLE IF EXISTS `' . _DB_PREFIX_ . 'smartsend_shipping`') &&
            Db::getInstance()->execute('DROP TABLE IF EXISTS `' . _DB_PREFIX_ . 'smartsend_agent`');
    }

    /*
     * Deactivate the Smartsend Label tab created when the module was installed
     * @return boolean
     */
    public function uninstallSmartsendLabelTab()
    {
        $idTab = Tab::getIdFromClassName('SmartsendLabel');
        if (!$idTab) {
            // No tab found
            return false;
        }

        while ($idTab) {
            $tab = new Tab($idTab);
            $tab->delete();

            $idTab = Tab::getIdFromClassName('SmartsendLabel');
        }

        return true;
    }

    /*
     * Deactivate the Smartsend Combine Label tab created when the module was installed
     * @return boolean
     */
    public function uninstallSmartsendCombineLabelTab()
    {
        $idTab = Tab::getIdFromClassName('SmartsendCombineLabel');
        if (!$idTab) {
            // No tab found
            return false;
        }

        while ($idTab) {
            $tab = new Tab($idTab);
            $tab->delete();

            $idTab = Tab::getIdFromClassName('SmartsendCombineLabel');
        }

        return true;
    }

    /*
     * Deactivate the Smartsend Action tab created when the module was installed
     * @return boolean
     */
    public function uninstallSmartsendActionTab()
    {
        $idTab = Tab::getIdFromClassName('SmartsendAction');
        if (!$idTab) {
            // No tab found
            return false;
        }

        while ($idTab) {
            $tab = new Tab($idTab);
            $tab->delete();

            $idTab = Tab::getIdFromClassName('SmartsendAction');
        }

        return true;
    }

    /*
     * Deactivate the Smartsend Shipping tab created when the module was installed
     * @return boolean
     */
    public function uninstallSmartsendShippingTab()
    {
        $idTab = Tab::getIdFromClassName('SmartsendShipping');
        if (!$idTab) {
            // No tab found
            return false;
        }

        while ($idTab) {
            $tab = new Tab($idTab);
            $tab->delete();

            $idTab = Tab::getIdFromClassName('SmartsendShipping');
        }

        return true;
    }

    /**
     * Load the configuration form
     */
    public function getContent()
    {
        $html = '';
        /**
         * If values have been submitted in the form, process.
         */
        if (((bool)Tools::isSubmit('submitSmartsendModule')) == true) {
            $this->postProcess();
            if (isset($this->errors)) {
                $this->context->controller->errors = $this->errors;
            }
            if (isset($this->confirmations)) {
                $this->context->controller->confirmations = $this->confirmations;
            }
            $url = $this->context->link->getAdminLink('AdminModules', false)
                . '&configure=' . $this->name . '&token=' . Tools::getAdminTokenLite('AdminModules');
        } else {
            $url = $this->context->link->getAdminLink('SmartsendShipping', false)
                . '&configure=' . $this->name . '&token=' . Tools::getAdminTokenLite('SmartsendShipping');
        }
        $this->context->smarty->assign('return_methods', $this->returnShippingMethods());
        $this->context->smarty->assign('allowed_shipping_methods', $this->allowedShippingMethods());
        $this->context->smarty->assign('ajax_url', $url);
        $this->context->smarty->assign('module_dir', $this->_path);
        $output = $this->context->smarty->fetch($this->local_path . 'views/templates/admin/configure.tpl');

        $html = $output . $this->renderForm();

        $content = $this->getListContent((int)Configuration::get('PS_LANG_DEFAULT'));
        $helper = $this->initList();
        $helper->listTotal = count($content);
        return $html . $helper->generateList($content, $this->fields_list);
    }

    /**
     * Create the form that will be displayed in the configuration of your module.
     */
    protected function renderForm()
    {
        $helper = new HelperForm();

        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = $this->context->language->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitSmartsendModule';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
            . '&configure=' . $this->name . '&tab_module=' . $this->tab . '&module_name=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFormValues(), /* Add values for your inputs */
            'languages'    => $this->context->controller->getLanguages(),
            'id_language'  => $this->context->language->id,
        );
        return $helper->generateForm($this->getConfigForm());
    }

    /**
     * Create the structure of your form.
     */
    protected function getConfigForm()
    {
        $psStates = OrderState::getOrderStates($this->context->language->id);
        $states = array(
            array(
                'value' => '',
                'name'  => $this->l('Do not change order status'),
            ),
        );
        foreach ($psStates as $state) {
            $states[] = array(
                'value' => $state['id_order_state'],
                'name'  => $state['name'],
            );
        }

        return array(
            'general_settings' => array(
                'form' => array(
                    'legend' => array(
                        'title' => $this->l('General Settings'),
                        'icon'  => 'icon-cogs',
                    ),
                    'input'  => array(
                        array(
                            'type'    => 'switch',
                            'label'   => $this->l('Live mode'),
                            'name'    => 'SMARTSEND_LIVE_MODE',
                            'is_bool' => true,
                            'desc'    => $this->l('Use this module in live mode'),
                            'values'  => array(
                                array(
                                    'id'    => 'active_on',
                                    'value' => true,
                                    'label' => $this->l('Enabled'),
                                ),
                                array(
                                    'id'    => 'active_off',
                                    'value' => false,
                                    'label' => $this->l('Disabled'),
                                ),
                            ),
                        ),
                        array(
                            'col'    => 6,
                            'type'   => 'text',
                            'prefix' => '<i class="icon icon-key"></i>',
                            'name'   => 'SMARTSEND_API_TOKEN',
                            'label'  => $this->l('API Token'),
                            'desc'   => $this->l('Enter a valid API Token'),
                        ),
                    ),
                    'submit' => array(
                        'title' => $this->l('Save and validate API Token'),
                    ),
                ),
            ),
            'label_settings'   => array(
                'form' => array(
                    'legend' => array(
                        'title' => $this->l('Shipping Label Settings'),
                    ),
                    'input'  => array(
                        array(
                            'label'      => $this->l('Set Order status after label print'),
                            'name'       => 'SMARTSEND_ORDER_STATUS',
                            'validation' => 'isInt',
                            'col'        => 6,
                            'cast'       => 'intval',
                            'type'       => 'select',
                            'options'    => array(
                                'query' => $states,
                                'id'    => 'value',
                                'name'  => 'name',
                            ),
                        ),
                        array(
                            'type'    => 'switch',
                            'label'   => $this->l('Add private order message'),
                            'name'    => 'SMARTSEND_ORDER_MESSAGE',
                            'is_bool' => true,
                            'values'  => array(
                                array(
                                    'id'    => 'active_on',
                                    'value' => true,
                                    'label' => $this->l('Yes'),
                                ),
                                array(
                                    'id'    => 'active_off',
                                    'value' => false,
                                    'label' => $this->l('No'),
                                ),
                            ),
                        ),
                        array(
                            'type'    => 'switch',
                            'label'   => $this->l('Include order comment on label'),
                            'name'    => 'SMARTSEND_ORDER_COMMENT_LABEL',
                            'is_bool' => true,
                            'values'  => array(
                                array(
                                    'id'    => 'active_on',
                                    'value' => true,
                                    'label' => $this->l('Enabled'),
                                ),
                                array(
                                    'id'    => 'active_off',
                                    'value' => false,
                                    'label' => $this->l('Disabled'),
                                ),
                            ),
                        ),
                        array(
                            'type'    => 'switch',
                            'label'   => $this->l('Send In Transit email'),
                            'desc'    => $this->l('Send PrestaShop email to customer with tracking information when a label has been generated'),
                            'name'    => 'SMARTSEND_SHIPMENT_EMAIL',
                            'is_bool' => true,
                            'values'  => array(
                                array(
                                    'id'    => 'active_on',
                                    'value' => true,
                                    'label' => $this->l('Enabled'),
                                ),
                                array(
                                    'id'    => 'active_off',
                                    'value' => false,
                                    'label' => $this->l('Disabled'),
                                ),
                            ),
                        ),
                    ),
                    'submit' => array(
                        'title' => $this->l('Save'),
                    ),
                ),
            ),
            'pickup_settings'  => array(
                'form' => array(
                    'legend' => array(
                        'title' => $this->l('Pickup Points Settings'),
                    ),
                    'input'  => array(
                        array(
                            'label'      => $this->l('DropDown Format'),
                            'name'       => 'SMARTSEND_DROPDOWN_FORMAT',
                            'validation' => 'isInt',
                            'cast'       => 'intval',
                            'col'        => 6,
                            'type'       => 'select',
                            'options'    => array(
                                'query' => array(
                                    array(
                                        'value' => 'company, street, postal_code, city',
                                        'name'  => '#Company, #Street, #Zipcode #City',
                                    ),
                                    array(
                                        'value' => 'company, street, postal_code',
                                        'name'  => '#Company, #Street, #Zipcode',
                                    ),
                                    array(
                                        'value' => 'company, street, city',
                                        'name'  => '#Company, #Street, #City',
                                    ),
                                    array(
                                        'value' => 'company, street',
                                        'name'  => '#Company, #Street',
                                    ),
                                ),
                                'id'    => 'value',
                                'name'  => 'name',
                            ),
                        ),
                        array(
                            'type'    => 'switch',
                            'label'   => $this->l('Auto-select closest'),
                            'name'    => 'SMARTSEND_AUTO_SELECT_CLOSEST',
                            'is_bool' => true,
                            'values'  => array(
                                array(
                                    'id'    => 'active_on',
                                    'value' => true,
                                    'label' => $this->l('Enabled'),
                                ),
                                array(
                                    'id'    => 'active_off',
                                    'value' => false,
                                    'label' => $this->l('Disabled'),
                                ),
                            ),
                        ),
                    ),
                    'submit' => array(
                        'title' => $this->l('Save'),
                    ),
                ),
            ),
        );
    }

    /**
     * This function returns all shipping methods on the grid list.
     */
    protected function getListContent()
    {
        $smartsend_method = new SmartSendMethodsClass();
        $smartsend_methods = $smartsend_method->getAllShippingMethods();
        return $smartsend_methods;
    }

    protected function initList()
    {
        $this->fields_list = array(
            'id_carrier'      => array(
                'title'   => $this->l('ID'),
                'width'   => 50,
                'type'    => 'text',
                'search'  => true,
                'orderby' => false,
                'align'   => 'center',
            ),
            'shipping_method' => array(
                'title'   => $this->l('Shipping Method'),
                'width'   => 120,
                'type'    => 'select_category',
                'search'  => false,
                'orderby' => false,
                'align'   => 'center',
                'list'    => $this->allowedShippingMethods(),
            ),
            'return_method'   => array(
                'title'   => $this->l('Return Method'),
                'width'   => 120,
                'type'    => 'select_category',
                'list'    => $this->returnShippingMethods(),
                'search'  => false,
                'orderby' => false,
                'align'   => 'center',
            ),
            'name'            => array(
                'title'   => $this->l('Name'),
                'width'   => 120,
                'type'    => 'text',
                'search'  => true,
                'orderby' => false,
            ),
            'image'           => array(
                'title'   => $this->l('Logo'),
                'align'   => 'center',
                'image'   => 's',
                'class'   => 'fixed-width-xs',
                'orderby' => false,
                'search'  => false,
            ),
            'delay'           => array(
                'title'   => $this->l('Delay'),
                'width'   => 120,
                'type'    => 'text',
                'search'  => true,
                'orderby' => false,
            ),
            'active'          => array(
                'title'   => $this->l('Active'),
                'align'   => 'center',
                'active'  => 'status',
                'type'    => 'bool',
                'class'   => 'fixed-width-sm',
                'orderby' => false,
            ),
            'current_version' => array(
                'title'   => $this->l('Current Version'),
                'align'   => 'center',
                'active'  => 'status',
                'type'    => 'bool',
                'class'   => 'fixed-width-sm',
                'orderby' => false,
            ),
        );

        $helper = new HelperList();
        $helper->shopLinkType = '';
        $helper->simple_header = false;
        $helper->identifier = 'id_carrier';
        // $helper->actions = array('edit', 'delete');
        $helper->show_toolbar = true;
        $helper->imageType = 'jpg';
        $helper->toolbar_btn['new'] = array(
            'href' => $this->context->link->getAdminLink('AdminCarrierWizard'),
            'desc' => $this->l('Add new'),
        );
        $helper->module = $this;
        $helper->title = 'SHIPPING METHODS';
        $helper->table = 'carrier';
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
            . '&configure=' . $this->name;
        return $helper;
    }

    /**
     * Set values for the inputs.
     */
    protected function getConfigFormValues()
    {
        return array(
            'SMARTSEND_LIVE_MODE'           => Configuration::get('SMARTSEND_LIVE_MODE', null),
            'SMARTSEND_API_TOKEN'           => Configuration::get('SMARTSEND_API_TOKEN', null),
            'SMARTSEND_ORDER_STATUS'        => Configuration::get('SMARTSEND_ORDER_STATUS', null),
            'SMARTSEND_ORDER_MESSAGE'       => Configuration::get('SMARTSEND_ORDER_MESSAGE', null),
            'SMARTSEND_ORDER_COMMENT_LABEL' => Configuration::get('SMARTSEND_ORDER_COMMENT_LABEL', null),
            'SMARTSEND_SHIPMENT_EMAIL'      => Configuration::get('SMARTSEND_SHIPMENT_EMAIL', null),
            'SMARTSEND_DROPDOWN_FORMAT'     => Configuration::get('SMARTSEND_DROPDOWN_FORMAT', null),
            'SMARTSEND_AUTO_SELECT_CLOSEST' => Configuration::get('SMARTSEND_AUTO_SELECT_CLOSEST', null),
        );
    }

    /**
     * Save form data.
     */
    protected function postProcess()
    {
        $form_values = $this->getConfigFormValues();
        foreach (array_keys($form_values) as $key) {
            if ($key != 'SMARTSEND_LIVE_MODE' && $key != 'SMARTSEND_API_TOKEN') {
                Configuration::updateValue($key, Tools::getValue($key));
            }
        }

        // Update API Token and Live Mode settings
        $new_api_token = Tools::getValue('SMARTSEND_API_TOKEN');
        $new_live_mode = Tools::getValue('SMARTSEND_LIVE_MODE');

        $live_mode_updated = Configuration::get('SMARTSEND_LIVE_MODE', null) != $new_live_mode;
        $api_token_updated = Configuration::get('SMARTSEND_API_TOKEN', null) != $new_api_token;

        Configuration::updateValue('SMARTSEND_API_TOKEN', $new_api_token);
        if ($live_mode_updated || $api_token_updated) {
            $smartsend_api = new SmartSendShippingApi($new_api_token, !$new_live_mode, null);
            if ($smartsend_api->validateToken()) {
                Configuration::updateValue('SMARTSEND_LIVE_MODE', $new_live_mode);
                $this->confirmations[] = 'Valid API Token';
            } else {
                Configuration::updateValue('SMARTSEND_LIVE_MODE', 0);
                $this->errors[] = 'Invalid API Token: ' . $smartsend_api->getErrorString();
            }
        }
    }

    public function getOrderShippingCost($params, $shipping_cost)
    {
        if (Context::getContext()->customer->logged == true) {
            $id_address_delivery = Context::getContext()->cart->id_address_delivery;
            $address = new Address($id_address_delivery);

            /**
             * Send the details through the API
             * Return the price sent by the API
             */
            return ($address && $params) ? 10 : 100;
        }

        return $shipping_cost;
    }

    public function getOrderShippingCostExternal($params)
    {
        return ($params ? true : false);
    }

    protected function addCarrier()
    {
        $carrier = new Carrier();

        $carrier->name = $this->l('My super carrier');
        $carrier->is_module = true;
        $carrier->active = 1;
        $carrier->range_behavior = 1;
        $carrier->need_range = 1;
        $carrier->shipping_external = true;
        $carrier->range_behavior = 0;
        $carrier->external_module_name = $this->name;
        $carrier->shipping_method = 2;

        foreach (Language::getLanguages() as $lang) {
            $carrier->delay[$lang['id_lang']] = $this->l('Super fast delivery');
        }

        if ($carrier->add() == true) {
            @copy(
                dirname(__FILE__) . '/views/img/carrier_image.jpg',
                _PS_SHIP_IMG_DIR_ . '/' . (int)$carrier->id . '.jpg'
            );
            Configuration::updateValue('MYSHIPPINGMODULE_CARRIER_ID', (int)$carrier->id);
            return $carrier;
        }

        return false;
    }

    protected function addGroups($carrier)
    {
        $groups_ids = array();
        $groups = Group::getGroups(Context::getContext()->language->id);
        foreach ($groups as $group) {
            $groups_ids[] = $group['id_group'];
        }

        $carrier->setGroups($groups_ids);
    }

    protected function addRanges($carrier)
    {
        $range_price = new RangePrice();
        $range_price->id_carrier = $carrier->id;
        $range_price->delimiter1 = '0';
        $range_price->delimiter2 = '10000';
        $range_price->add();

        $range_weight = new RangeWeight();
        $range_weight->id_carrier = $carrier->id;
        $range_weight->delimiter1 = '0';
        $range_weight->delimiter2 = '10000';
        $range_weight->add();
    }

    protected function addZones($carrier)
    {
        $zones = Zone::getZones();

        foreach ($zones as $zone) {
            $carrier->addZone($zone['id_zone']);
        }
    }

    /**
     * Add the CSS & JavaScript files you want to be loaded in the BO.
     */
    public function hookDisplayBackOfficeHeader()
    {
        if (Tools::getValue('configure') == $this->name) {
            $this->context->controller->addJS($this->_path . 'views/js/jquery.min.js');
            $this->context->controller->addJS($this->_path . 'views/js/back.js');
            $this->context->controller->addCSS($this->_path . 'views/css/back.css');
        }
    }

    /**
     * Add the CSS & JavaScript files you want to be added on the FO.
     */
    public function hookHeader()
    {
        // $this->context->controller->addJS($this->_path . '/views/js/jquery/jquery-2.1.4.min.js');
        $this->context->controller->addJS($this->_path . '/views/js/front.js');
        $this->context->controller->addCSS($this->_path . '/views/css/front.css');
    }

    public function hookActionCarrierUpdate($params)
    {
        $id_carrier_old = (int)($params['id_carrier']);
        $id_carrier_new = (int)($params['carrier']->id);

        $smartsend_shipping = new SmartSendMethodsClass();
        $smartsend_carrier_id = $smartsend_shipping->getShippingMethodByCarrierId($id_carrier_old);

        $smartsend_old_data = new SmartSendMethodsClass($smartsend_carrier_id);

        $data = array();

        $data['id_carrier'] = $id_carrier_new;
        $data['shipping_method'] = $smartsend_old_data->shipping_method;
        $data['return_method'] = $smartsend_old_data->return_method;
        $data['current_version'] = $smartsend_old_data->current_version;

        $smartsend_shipping->copyFromPost($data);
        if ($smartsend_shipping->validateFields(false)) {
            $smartsend_shipping->save();
        }
    }

    /**
     * This function returns only return shipping methods.
     */
    protected function returnShippingMethods()
    {
        return array(
            '0'        => $this->l('- Select Return Method -'),
            'PostNord' =>
                array(
                    'postnord_returndropoff' => $this->l('PostNord: Return from pick-up point (Return Drop Off)'),
                    'postnord_returnpickup'  => $this->l('PostNord: Return from address (Return Pickup)'),
                ),
            'GLS'      =>
                array(
                    'gls_returndropoff' => $this->l('GLS: Return from pick-up point (ShopReturn)'),
                ),
            'DAO'      =>
                array(
                    'dao_returndropoff' => $this->l('DAO: Return from pick-up point (ParcelShop Return)'),
                ),
            'Bring'    =>
                array(
                    'bring_returndropoff' => $this->l('Bring: Return from pick-up point (PickUp Parcel Return)'),
                    'bring_returnpickup'  => $this->l('Bring: Return from address (Parcel Return)'),
                ),
        );
    }

    /**
     * This function returns all shipping methods.
     */
    protected function allowedShippingMethods()
    {
        return array(
            '0'                 => $this->l('- Select Shipping Method -'),
            'PostNord'          =>
                array(
                    'postnord_agent'                   => $this->l('PostNord: Select pick-up point (MyPack Collect)'),
                    'postnord_collect'                 => $this->l('PostNord: Closest pick-up point (MyPack Collect)'),
                    'postnord_homedelivery'            => $this->l(
                        'PostNord: Private delivery to address (MyPack Home)'
                    ),
                    'postnord_doorstep'                => $this->l('PostNord: Leave at door (Flexdelivery)'),
                    'postnord_flexhome'                => $this->l('PostNord: Flexible home delivery (FlexChange)'),
                    'postnord_homedeliveryeconomy'     => $this->l(
                        'PostNord: Private economy delivery to address (MyPack Home Economy)'
                    ),
                    'postnord_homedeliverysmall'       => $this->l(
                        'PostNord: Private delivery to address Small (MyPack Home Small)'
                    ),
                    'postnord_commercial'              => $this->l('PostNord: Commercial delivery to address (Parcel)'),
                    'postnord_valuableparcel'          => $this->l('PostNord: Valuable parcel'),
                    'postnord_valuemaillarge'          => $this->l('PostNord: Tracked Valuemail Large'),
                    'postnord_valuemailmaxi'           => $this->l('PostNord: Tracked Valuemail Maxi'),
                    'postnord_valuemailfirstclass'     => $this->l('PostNord: Tracked Valuemail First Class'),
                    'postnord_valuemaileconomy'        => $this->l('PostNord: Tracked Valuemail Economy'),
                    'postnord_valuemaileco'            => $this->l('PostNord: Tracked Valuemail Eco Friendly'),
                    'postnord_valuemailuntrackedlarge' => $this->l('PostNord: Untracked Valuemail Large'),
                    'postnord_valuemailuntrackedmaxi'  => $this->l('PostNord: Untracked Valuemail Maxi'),
                    'postnord_letterregistered'        => $this->l('PostNord: Registred letter'),
                    'postnord_lettertracked'           => $this->l('PostNord: Tracked letter'),
                    'postnord_letteruntracked'         => $this->l('PostNord: Untracked letter'),
                ),
            'GLS'               =>
                array(
                    'gls_agent'        => $this->l('GLS: Select pick-up point (ParcelShop)'),
                    'gls_collect'      => $this->l('GLS: Closest pick-up point (ParcelShop)'),
                    'gls_homedelivery' => $this->l('GLS: Private delivery to address (PrivateDelivery)'),
                    'gls_doorstep'     => $this->l('GLS: Leave at door (DepositService)'),
                    'gls_flexhome'     => $this->l('GLS: Flexible home delivery (FlexDelivery)'),
                    'gls_commercial'   => $this->l('GLS: Commercial delivery to address (BusinessParcel)'),
                ),
            'DAO'               =>
                array(
                    'dao_agent'    => $this->l('DAO: Select pick-up point (ParcelShop)'),
                    'dao_collect'  => $this->l('DAO: Closest pick-up point (ParcelShop)'),
                    'dao_doorstep' => $this->l('DAO: Leave at door (Direct)'),
                ),
            'Bring'             =>
                array(
                    'bring_agent'                   => $this->l('Bring: Select pick-up point
                    (PickUp Parcel / Serviceparcel)'),
                    'bring_bulkagent'               => $this->l('Bring: Select pick-up point, send as bulk
                    (PickUp Parcel Bulk)'),
                    'bring_collect'                 => $this->l('Bring: Closest pick-up point
                    (PickUp Parcel / Serviceparcel)'),
                    'bring_bulkcollect'             => $this->l('Bring: Closest pick-up point, send as bulk
                    (PickUp Parcel Bulk)'),
                    'bring_homedelivery'            => $this->l('Bring: Private delivery to address
                    (Home Delivery Parcel)'),
                    'bring_commercial'              => $this->l('Bring: Commercial delivery to address
                    (Business Parcel)'),
                    'bring_bulkcommercial'          => $this->l(
                        'Bring: Commercial delivery to address, send as bulk (Business Parcel Bulk)'
                    ),
                    'bring_commercialfullpallet'    => $this->l(
                        'Bring: Commercial delivery of full size pallet (Business Pallet)'
                    ),
                    'bring_commercialhalfpallet'    => $this->l(
                        'Bring: Commercial delivery of half size pallet (Business Pallet)'
                    ),
                    'bring_commercialquarterpallet' => $this->l(
                        'Bring: Commercial delivery of quarter size pallet (Business Pallet)'
                    ),
                    'bring_express9'                => $this->l(
                        'Bring: Express delivery before 9:00 (Express Nordic 09:00)'
                    ),
                    'bring_bulkexpress9'            => $this->l(
                        'Bring: Express delivery before 9:00, send as bulk (Express Nordic 09:00 Bulk)'
                    ),
                ),
            'Bifrost Logistics' =>
                array(
                    // eTail Tracked
                    'bifrost_etailtracked'            => $this->l('Bifrost Logistics: eTail Tracked'),
                    'bifrost_etailtrackedlarge'       => $this->l('Bifrost Logistics: eTail Tracked large'),
                    // eTail Gain
                    'bifrost_etailgainsmall'          => $this->l('Bifrost Logistics: eTail Gain small'),
                    'bifrost_etailgainlarge'          => $this->l('Bifrost Logistics: eTail Gain large'),
                    // Letter Priority
                    'bifrost_letterprioritysmall'     => $this->l('Bifrost Logistics: Letter priority small'),
                    'bifrost_letterprioritylarge'     => $this->l('Bifrost Logistics: Letter priority large'),
                    'bifrost_letterprioritymaxi'      => $this->l('Bifrost Logistics: Letter priority maxi'),
                    //Letter Economy
                    'bifrost_lettereconomysmall'      => $this->l('Bifrost Logistics: Letter economy small'),
                    'bifrost_lettereconomylarge'      => $this->l('Bifrost Logistics: Letter economy large'),
                    'bifrost_lettereconomymaxi'       => $this->l('Bifrost Logistics: Letter economy maxi'),
                    // Press Priority
                    'bifrost_pressprioritylarge'      => $this->l('Bifrost Logistics: Press priority large'),
                    'bifrost_pressprioritymaxi'       => $this->l('Bifrost Logistics: Press priority maxi'),
                    // Advertising Economy
                    'bifrost_advertisingeconomysmall' => $this->l('Bifrost Logistics: Advertising economy small'),
                    'bifrost_advertisingeconomylarge' => $this->l('Bifrost Logistics: Advertising economy large'),
                    'bifrost_advertisingeconomymaxi'  => $this->l('Bifrost Logistics: Advertising economy maxi'),
                    // Ecom priority large
                    'bifrost_ecomprioritylarge'       => $this->l('Bifrost Logistics: Ecom priority large'),
                    'bifrost_ecomprioritymaxi'        => $this->l('Bifrost Logistics: Ecom priority maxi'),
                ),
        );
    }

    private function moveMessagesFromCookieToContext()
    {
        $context = Context::getContext();
        if (Tools::getIsset('generate_action')) {
            $smartsend_action_errors = json_decode($context->cookie->smartsend_action_errors);
            $smartsend_action_warnings = json_decode($context->cookie->smartsend_action_warnings);
            $smartsend_action_confirmations = json_decode($context->cookie->smartsend_action_confirmations);
            $smartsend_action_informations = json_decode($context->cookie->smartsend_action_informations);

            if (is_array($smartsend_action_errors)) {
                foreach ($smartsend_action_errors as $error) {
                    $context->controller->errors[] = $error;
                }
            }

            if (is_array($smartsend_action_warnings)) {
                foreach ($smartsend_action_warnings as $warning) {
                    $context->controller->warning[] = $warning;
                }
            }

            if (is_array($smartsend_action_confirmations)) {
                foreach ($smartsend_action_confirmations as $confirmations) {
                    $context->controller->confirmations[] = $confirmations;
                }
            }

            if (is_array($smartsend_action_informations)) {
                foreach ($smartsend_action_informations as $informations) {
                    $context->controller->informations[] = $informations;
                }
            }

            //$context->cookie->__set('smartsend_action_errors', null);
            //$context->cookie->__set('smartsend_action_warnings', null);
            //$context->cookie->__set('smartsend_action_confirmations', null);
            //$context->cookie->__set('smartsend_action_informations', null);
        }
    }

	//Hook to add success message from version 1.7.0
	public function hookDisplayAdminAfterHeader($params)
    {
        $context = Context::getContext();
		$content = '';

        if (Tools::getIsset('generate_action')) {
			$smartsend_action_errors = json_decode($context->cookie->smartsend_action_errors);
            $smartsend_action_warnings = json_decode($context->cookie->smartsend_action_warnings);
            $smartsend_action_confirmations = json_decode($context->cookie->smartsend_action_confirmations);
            $smartsend_action_informations = json_decode($context->cookie->smartsend_action_informations);

			$confirmationsTexts = array();
			$errorTexts = array();
			$warningTexts = array();
			$infoTexts = array();

            if (is_array($smartsend_action_errors)) {
                foreach ($smartsend_action_errors as $error) {
                    $errorTexts[] = $error;
                }
            }

            if (is_array($smartsend_action_warnings)) {
                foreach ($smartsend_action_warnings as $warning) {
                    $warningTexts[] = $warning;
                }
            }

            if (is_array($smartsend_action_informations)) {
                foreach ($smartsend_action_informations as $informations) {
                    $infoTexts[] = $informations;
                }
            }

            if (is_array($smartsend_action_confirmations)) {
                foreach ($smartsend_action_confirmations as $confirmations) {
                    $confirmationsTexts[] = $confirmations;
                }
            }

			$content = '';

			if(!empty($confirmationsTexts)) {
				$content .= '<div class="alert alert-success d-print-none" role="alert">
							  <button type="button" class="close" data-dismiss="alert" aria-label="Close"> <span aria-hidden="true"><i class="material-icons">close</i></span> </button>
							  <div class="alert-text">';
				foreach($confirmationsTexts as $confirmationsText) {
					$content .= '<p>'.$confirmationsText.'</p>';
				}

				$content .= '</div></div>';
			}

			if(!empty($errorTexts)) {
				$content .= '<div class="alert alert-danger d-print-none" role="alert">
							  <button type="button" class="close" data-dismiss="alert" aria-label="Close"> <span aria-hidden="true"><i class="material-icons">close</i></span> </button>
							  <div class="alert-text">';
				foreach($errorTexts as $errorText) {
					$content .= '<p>'.$errorText.'</p>';
				}

				$content .= '</div></div>';
			}

			if(!empty($warningTexts)) {
				$content .= '<div class="alert alert-warning d-print-none" role="alert">
							  <button type="button" class="close" data-dismiss="alert" aria-label="Close"> <span aria-hidden="true"><i class="material-icons">close</i></span> </button>
							  <div class="alert-text">';
				foreach($warningTexts as $warningText) {
					$content .= '<p>'.$warningText.'</p>';
				}

				$content .= '</div></div>';
			}
        }

		return $content;
    }

    /*
     * Hook to add label buttons at shipping block on order view page
     */
    public function hookDisplayAdminOrderMain(array $params)
    {
        include_once _PS_MODULE_DIR_ . 'smartsend/smartsendshipmentlabels.php';
        include_once _PS_MODULE_DIR_ . 'smartsend/classes/class.agent.php';

        $context = Context::getContext();
        $this->moveMessagesFromCookieToContext();

        $order_id = (int)Tools::getValue('id_order');
        $request_uri = Context::getContext()->link->getAdminLink('SmartsendAction') . "&id_order=" . $order_id;
        $id_lang = Context::getContext()->cart->id_lang;
        $order = new Order((int)$order_id);
        $tracking_url = null;

        $smartsend_method = new SmartSendMethodsClass();
        $rule_id = $smartsend_method->getShippingMethodByCarrierId($order->id_carrier);
        $shipping_rule_details = new SmartSendMethodsClass($rule_id);

        $is_pickup_method = false;
        if ($this->isAgentAllowed($shipping_rule_details->shipping_method)) {
            $is_pickup_method = true;
        }
        $smartsend_agent = new SmartSendAgentClass();
        $agent_details = $smartsend_agent->getAgentDetailsByOrderId($order_id);

        if ($order->shipping_number) {
            $shipping_labels = new SmartSendShipmentLabelsClass();

            $shipping_method_id = explode('_', $shipping_rule_details->shipping_method);
            $shipping_carrier = $shipping_method_id[0];
            $shipping_api = new SmartSendShippingApi;
            $tracking_url = $shipping_api->getDefaultTrackingGateway()
                . $shipping_carrier . '/' . $order->shipping_number;

            $shipment_label_details = $shipping_labels->getShipmentLabelDetailsByOrderId($order->id);
        }

        $context->smarty->assign(array(
            'id_lang'               => $id_lang,
            'request_uri'           => $request_uri,
            'is_order_page'         => false,
            'live_mode'             => Configuration::get('SMARTSEND_LIVE_MODE', null),
            'tracking_url'          => $tracking_url,
            'label_pdf_link'        => isset($shipment_label_details['pdf_link']) ?
                $shipment_label_details['pdf_link'] : null,
            'return_label_pdf_link' => isset($shipment_label_details['return_pdf_link']) ?
                $shipment_label_details['return_pdf_link'] : null,
            'is_pickup_method'      => $is_pickup_method,
            'agent_data'           => $agent_details,
        ));

        $content = $context->smarty->fetch(_PS_MODULE_DIR_ . '/smartsend/views/templates/hook/content_ship.tpl');

        return $content;
    }


	/*
     * Hook to add label buttons at order bulk options
     */
    public function hookDisplayBackOfficeFooter($params)
    {
        if (Tools::getValue('controller') == 'AdminOrders' && !Tools::getValue('id_order')) {
            $request_uri = Context::getContext()->link->getAdminLink('SmartsendAction');
            $context = Context::getContext();
            $id_lang = $context->cookie->id_lang;

            $this->moveMessagesFromCookieToContext();

            $context->smarty->assign(array(
                'request_uri'   => $request_uri,
                'id_lang'       => $id_lang,
                'is_order_page' => true,
                'live_mode'     => Configuration::get('SMARTSEND_LIVE_MODE', null),
            ));
            $content = $context->smarty->fetch(_PS_MODULE_DIR_ . '/smartsend/views/templates/hook/label_buttons.tpl');

            return $content;
        }
    }

    public function hookActionCarrierProcess($params)
    {
        include_once _PS_MODULE_DIR_ . 'smartsend/classes/class.agent.php';
        $data = Tools::getAllValues();
        $pickup_point = null;
        $pickup_data = array();
        $pickup = array();

        $context = Context::getContext();
        if (isset($data['delivery_option'])) {
            $delivery_option = $data['delivery_option'];
            foreach ($delivery_option as $value) {
                $carrier_id = str_replace(',', '', $value);
                $smartsend_agent = new SmartSendAgentClass();
                if (isset($data['shipping_agent_'.$carrier_id])) {
                    $pickup_point = $data['shipping_agent_'.$carrier_id];
                    $pickup[$carrier_id] = Tools::jsonDecode(Tools::htmlentitiesDecodeUTF8($pickup_point), true);
                    if (!empty($pickup)) {
                        foreach ($pickup as $carrier_id => $pickup_point) {
                            $pickup_data['id_carrier'] = $carrier_id;
                            $pickup_data['cart_id'] = $params['cart']->id;
                            $pickup_data['store'] = Configuration::get('PS_SHOP_NAME');
                            $pickup_data['agent_no'] = $pickup_point['agent_no'];
                            $pickup_data['company'] = $pickup_point['company'];
                            $pickup_data['address_line1'] = $pickup_point['address_line1'];
                            $pickup_data['address_line2'] = $pickup_point['address_line2'];
                            $pickup_data['city'] = $pickup_point['city'];
                            $pickup_data['postal_code'] = $pickup_point['postal_code'];
                            $pickup_data['country'] = $pickup_point['country'];
                        }
                    }
                    if (!empty($pickup_data)) {
                        if ($agent_details = $smartsend_agent->getAgentDetailsByCartId($params['cart']->id)) {
                            $pickup_data['id'] = $agent_details['id'];
                        }
                        $smartsend_agent->copyFromPost($pickup_data);
                        if ($smartsend_agent->validateFields(false)) {
                            $smartsend_agent->save();
                        }
                    }
                } elseif ($smartsend_agent->getAgentDetailsByCartId($params['cart']->id)) {
                    $smartsend_agent->deleteAgentDetails($params['cart']->id);
                }
            }
        }
    }

    public function hookDisplayBeforeCarrier($params)
    {
        return $this->showPickupPoints($params);
    }

    private function showPickupPoints($params)
    {
        $jquery_include = false;
        $version = '1.7.0.0';
        if (version_compare(_PS_VERSION_, $version) < 0) {
            $jquery_include = false;
        } else {
            $jquery_include = true;
        }
        $cart = $params['cart'];
        $address = new Address($cart->id_address_delivery);
        $country = new Country($address->id_country);

        $carriers = Carrier::getCarriers($cart->id_lang, false, false, false, null, Carrier::ALL_CARRIERS);

        $html = '';
        foreach ($carriers as $carrier) {
            $id = $carrier['id_carrier'];
            $smartsend_method = new SmartSendMethodsClass();
            $rule_id = $smartsend_method->getShippingMethodByCarrierId($id);

            if ($rule_id) {
                $shipping_rule_details = new SmartSendMethodsClass($rule_id);

                if (!empty($shipping_rule_details->shipping_method)
                    && $this->isAgentAllowed($shipping_rule_details->shipping_method)) {
                    $shipping_method = $shipping_rule_details->shipping_method;
                    $street = $address->address1;
                    $postal_code = $address->postcode;
                    $city = $address->city;
                    $iso_code = $country->iso_code;

                    $pickup_available = false;
                    $pickup_points = $this->getPickupPoints($shipping_method, $iso_code, $postal_code, $city, $street);
                    if (!empty($pickup_points)) {
                        $pickup_available = true;
                    }
                    $this->context->smarty->assign(
                        array(
                            'pickup_points' => json_decode(json_encode($pickup_points), true),//TODO: Fix this encoding+decoding
                            'carrierId' => $id,
                            'moduleName' => Tools::strtoupper($this->name),
                            'dropdown_format' => Configuration::get('SMARTSEND_DROPDOWN_FORMAT', null),
                            'selectError' => $this->l('You must select a Pickup point'),
                            'jquery_include' => $jquery_include,
                            'pickup_available' => $pickup_available,
                            'auto_select_pickup' => Configuration::get('SMARTSEND_AUTO_SELECT_CLOSEST', null)
                        )
                    );
                    $html .= $this->context->smarty
                        ->fetch(_PS_MODULE_DIR_ . '/smartsend/views/templates/hook/pickup_point.tpl');
                }
            }
        }
        return $html;
    }

    private function getPickupPoints($carrier, $country, $postal_code, $city, $street)
    {
        $carrier = str_replace('_agent', '', $carrier);
        $smartsend_api = new SmartSendShippingApi();
        $response = $smartsend_api->getAgentPoints($carrier, $country, $postal_code, $city, $street);

        if (isset($response['data'])) {
            return $response['data'];
        } else {
            return false;
        }
    }

    private function isAgentAllowed($shipping_method)
    {
        if (strpos($shipping_method, 'agent') !== false) {
            return true;
        }
        return false;
    }

    public function getModuleLink($id_cart, $id_code, $name)
    {
        if ($id_cart && $id_code) {
            $params = array('id_cart' => $id_cart, 'id_code' => $id_code);
        } else {
            $params = array();
        }
        return $this->context->link->getModuleLink($this->name, $name, $params, true);
    }

    public function hookActionValidateOrder($params)
    {
        include_once _PS_MODULE_DIR_ . 'smartsend/classes/class.agent.php';
        $order = $params['order'];
        if (!Validate::isLoadedObject($order)) {
            return;
        }
        $smartsend_method = new SmartSendMethodsClass();
        $rule_id = $smartsend_method->getShippingMethodByCarrierId($order->id_carrier);

        if (!$rule_id) {
            return;
        }
        $shipping_rule_details = new SmartSendMethodsClass($rule_id);
        if (!empty($shipping_rule_details->shipping_method)
            && $this->isAgentAllowed($shipping_rule_details->shipping_method)) {
            $smartsend_agent = new SmartSendAgentClass();
            $agent_details = $smartsend_agent->getAgentDetailsByCartId($params['cart']->id);
            $data = array();
            if (!empty($agent_details)) {
                if ($agent_details['id_carrier'] == $order->id_carrier) {
                    $agent_details['order_id'] = $order->id;
                }

                if (!empty($agent_details)) {
                    $smartsend_agent->copyFromPost($agent_details);
                    if ($smartsend_agent->validateFields(false)) {
                        $smartsend_agent->save();
                    }
                }
            }
        }
        return true;
    }
}
