<?php
/**
 * LICENSE
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@smartsend.io so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade the plugin to newer
 * versions in the future. If you wish to customize the plugin for your
 * needs please refer to https://www.smartsend.io
 *
 * @author      Smart Send ApS
 * @copyright   Copyright (c) Smart Send ApS (https://www.smartsend.io)
 * @license     https://smartsend.io/license
 * @url         https://smartsend.io/
 * @class       SmartsendActionController
 * @folder      smartsend/controllers/admin/SmartsendActionController.php
 *
 */

class SmartsendActionController extends ModuleAdminController
{

    public function __construct()
    {
        parent::__construct();
        // If a single order button was pressed
        if (Tools::getIsset('smartsend_action')) {
            //Get the order ID from the URL and create an order object.
            $order_id = (int)Tools::getValue('id_order');
            $smartsend_action = Tools::getValue('smartsend_action');
            if (!$order_id) {
                $order_id = Tools::getValue('orderIds');
                $smartsend_action = 'bulk_' . $smartsend_action;
            }

            if (!Configuration::get('SMARTSEND_API_TOKEN', null)) {
                $this->addMessage(
                    'error',
                    'API Token required. Please enter API token in the module settings'
                );
            } else {
                switch ($smartsend_action) {
                    case 'generate_label':
                        $this->createLabelAction($order_id);
                        break;
                    case 'generate_return_label':
                        $this->createReturnLabelAction($order_id);
                        break;
                    case 'bulk_generate_label':
                        $this->createMultipleLabelsAction($order_id);
                        break;
                    case 'bulk_generate_return_label':
                        $this->createMultipleReturnLabelsAction($order_id);
                        break;
                }
            }
        }
        $url = $_SERVER['HTTP_REFERER'];
        Tools::redirectAdmin($url . "&generate_action=1");
    }

    private function addAllMessages()
    {
        $context = Context::getContext();

		$this->removeMessages('error');
        $this->removeMessages('warning');
        $this->removeMessages('confirmation');
        $this->removeMessages('information');

		$this->addMessage('error', $context->controller->errors);
        $this->addMessage('warning', $context->controller->warnings);
        $this->addMessage('confirmation', $context->controller->confirmations);
        $this->addMessage('information', $context->controller->informations);
    }

    /*
     * Add a message or array of messages to the cookie
     *
     * @param string $type can be either error, warning, confirmation or information
     * @param string|array $message
     */
    private function addMessage($type, $message)
    {
        switch ($type) {
            case 'error':
                $cookie = 'smartsend_action_errors';
                break;
            case 'warning':
                $cookie = 'smartsend_action_warnings';
                break;
            case 'confirmation':
                $cookie = 'smartsend_action_confirmations';
                break;
            case 'information':
                $cookie = 'smartsend_action_informations';
                break;
            default:
                throw new Exception('Unknown type');
        }

        if (!empty($message)) {

            $context = Context::getContext();

            $existing_messages = $context->cookie->{$cookie};
            if ($existing_messages) {
                $existing_messages = json_decode($existing_messages, true);
            } else {
                $existing_messages = array();
            }

            $all_messages = array(); // Changed array value by Shardul to show only one message - original was : $existing_messages;

            if (is_array($message)) {
                $i=-count($message);
                foreach ($message as $msg) {
                    $i++;
                    if ($i ==0 && $type == 'confirmation') {
                        $all_messages[] = $msg . '<br>';
                    } else {
                        $all_messages[] = $msg;
                    }
                }
            } else {
                $all_messages[] = $message;
            }

            if (!empty($all_messages)) {
                $context->cookie->__set($cookie, json_encode($all_messages));
            }
        }

    }

    /*
     * Add a message or array of messages to the cookie
     *
     * @param string $type can be either error, warning, confirmation or information
     * @param string|array $message
     */
    private function removeMessages($type)
    {
        switch ($type) {
            case 'error':
                $cookie = 'smartsend_action_errors';
                break;
            case 'warning':
                $cookie = 'smartsend_action_warnings';
                break;
            case 'confirmation':
                $cookie = 'smartsend_action_confirmations';
                break;
            case 'information':
                $cookie = 'smartsend_action_informations';
                break;
            default:
                throw new Exception('Unknown type');
        }

        $context = Context::getContext();
        $context->cookie->__set($cookie, null);

    }

    /*
     * Function to generate a label
     */
    public function createLabelAction($order_id)
    {
        require_once(_PS_MODULE_DIR_ . '/smartsend/controllers/admin/SmartsendLabelController.php');

        $order = new Order((int)$order_id);
        if ($order->id) {
            $labelController = new SmartsendLabelController($order, false);
            $labelController->createShipmentAndGenerateLabel();
            $this->addAllMessages();
        }
        return true;
    }

    /*
     * Function to generate a return label
     */
    public function createReturnLabelAction($order_id)
    {
        require_once(_PS_MODULE_DIR_ . '/smartsend/controllers/admin/SmartsendLabelController.php');

        $order = new Order((int)$order_id);
        if ($order->id) {
            $type = 'return_label';
            $labelController = new SmartsendLabelController($order, true);
            $labelController->createShipmentAndGenerateLabel($type);
            $this->addAllMessages();
        }
        return true;
    }

    /*
     * Function to generate a bulk labels
     */
    public function createMultipleLabelsAction($order_ids)
    {
        require_once(_PS_MODULE_DIR_ . '/smartsend/controllers/admin/SmartsendLabelController.php');

        if (is_array($order_ids) && !empty($order_ids)) {
            if (count($order_ids) < 6) {
                $shipments_created = array();

                foreach ($order_ids as $order_id) {
                    $order = new Order((int)$order_id);
                    if ($order->id) {
                        $labelController = new SmartsendLabelController($order, false);
                        $shipment_id = $labelController->createShipmentAndGenerateLabel(false,$order_id);

                        if ($shipment_id) {
                            $shipments_created[] = $shipment_id;
                        }

                        $this->addAllMessages();
                    }
                }

                if (count($shipments_created) > 1) {
                    $this->combineLabelPdfs($shipments_created);
                }
            } else {
                $this->addMessage('error', 'Currently maximum 5 orders can be handled simultaneously. Contact Smart Send for upgrade to handle more orders.');
            }
        }
        return true;
    }

    /*
     * Function to generate a bulk return labels
     */
    public function createMultipleReturnLabelsAction($order_ids)
    {
        require_once(_PS_MODULE_DIR_ . '/smartsend/controllers/admin/SmartsendLabelController.php');

        if (is_array($order_ids) && !empty($order_ids)) {
            if (count($order_ids) < 6) {
                $shipments_created = array();

                foreach ($order_ids as $order_id) {
                    $order = new Order((int)$order_id);
                    if ($order->id) {
                        $labelController = new SmartsendLabelController($order, true);
                        $shipment_id = $labelController->createShipmentAndGenerateLabel(true,$order_id);

                        if ($shipment_id) {
                            $shipments_created[] = $shipment_id;
                        }

                        $this->addAllMessages();
                    }
                }

                if (count($shipments_created) > 1) {
                    $this->combineLabelPdfs($shipments_created);
                }
            } else {
                $this->addMessage('error', 'Currently maximum 5 orders can be handled simultaneously. Contact Smart Send for upgrade to handle more orders.');
            }
        }
        return true;
    }

    /*
     * Function to combine PDF labels for shipments
     */
    private function combineLabelPdfs($array_shipment_ids)
    {
        require_once(_PS_MODULE_DIR_ . '/smartsend/controllers/admin/SmartsendCombineLabelController.php');

        $labelController = new SmartsendCombineLabelController($array_shipment_ids);
        $combined = $labelController->combineLabelsForShipments();

        if ($combined->isSuccessful()) {
            $response = $combined->getData();
            $tmp_translated_success = 'Shipping labels created by Smart Send. <a href="%s" target="_blank">Download combined shipping label.</a>';

            $this->removeMessages('confirmation');//We have to remove the existing confirmation messages to avoid the final message to exceed the maximum length
            $this->addMessage('confirmation', sprintf($tmp_translated_success, $response->pdf->link));
        } else {
            $this->addMessage('error', 'Unable to combine shipping labels');
        }
    }
}
